<?php
/**
 * Weels API Client
 * Calls the Weels REST API using wp_remote_* with X-API-Key authentication.
 */

defined('ABSPATH') || exit;

class Weels_API
{
    private string $api_key;
    private string $base_url;

    public function __construct(?string $api_key = null)
    {
        $this->api_key  = $api_key ?: get_option('weels_api_key', '');
        $this->base_url = WEELS_API_BASE;
    }

    public function has_key(): bool
    {
        return !empty($this->api_key);
    }

    /**
     * Validate the API key by calling the rates endpoint with a known postal code.
     */
    public function validate_key(): array
    {
        $resp = $this->post('/shipment/rates', [
            'postal' => 'M5H2M9',
            'qty'    => 1,
        ]);

        if (is_wp_error($resp)) {
            return ['valid' => false, 'error' => $resp->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($resp);
        if ($code === 401) {
            return ['valid' => false, 'error' => 'Invalid API key'];
        }

        return ['valid' => $code === 200, 'error' => $code !== 200 ? 'Unexpected response (' . $code . ')' : null];
    }

    /**
     * Get shipping rates for a destination postal code.
     */
    public function get_rates(string $postal, int $qty = 1, array $options = []): ?array
    {
        $body = array_merge([
            'postal' => $postal,
            'qty'    => $qty,
        ], $options);

        $resp = $this->post('/shipment/rates', $body);
        if (is_wp_error($resp)) return null;

        $code = wp_remote_retrieve_response_code($resp);
        if ($code !== 200) return null;

        return json_decode(wp_remote_retrieve_body($resp), true);
    }

    /**
     * Create a shipment.
     * @return array{success: bool, task_id?: int, label_pdf?: string, tracking_url?: string, error?: string}
     */
    public function create_shipment(array $data): array
    {
        $resp = $this->post('/shipment/create', $data);

        if (is_wp_error($resp)) {
            return ['success' => false, 'error' => $resp->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = json_decode(wp_remote_retrieve_body($resp), true);

        if ($code === 200 || $code === 201) {
            return [
                'success'      => true,
                'task_id'      => (int) ($body['task_id'] ?? 0),
                'total_price'  => (float) ($body['total_price'] ?? 0),
                'tracking_url' => $body['tracking_url'] ?? ('https://www.weels.ca/track/' . ($body['task_id'] ?? '')),
                'label_pdf'    => $body['label_pdf'] ?? null,
                'label_zpl'    => $body['label_zpl'] ?? null,
            ];
        }

        return ['success' => false, 'error' => $body['error'] ?? 'Failed to create shipment (HTTP ' . $code . ')'];
    }

    /**
     * Void a shipment.
     */
    public function void_shipment(int $task_id): array
    {
        $resp = $this->post('/shipment/void', ['task_id' => $task_id]);

        if (is_wp_error($resp)) {
            return ['success' => false, 'error' => $resp->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = json_decode(wp_remote_retrieve_body($resp), true);

        return [
            'success' => $code === 200 && !empty($body['success']),
            'error'   => $body['error'] ?? null,
        ];
    }

    /**
     * Get label URL for a task.
     */
    public function get_label_url(int $task_id, string $format = 'pdf'): string
    {
        return $this->base_url . '/shipment/label?task_id=' . $task_id . '&format=' . $format;
    }

    /**
     * Get tracking URL for a task.
     */
    public function get_tracking_url(int $task_id): string
    {
        return 'https://www.weels.ca/track/' . $task_id;
    }

    private function post(string $endpoint, array $body): \WP_Error|array
    {
        return wp_remote_post($this->base_url . $endpoint, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded',
                'X-API-Key'    => $this->api_key,
            ],
            'body' => $body,
        ]);
    }
}
