<?php
/**
 * Weels Order Meta Box
 * Adds a "Weels Delivery" panel on the order edit screen with label/track/void actions.
 */

defined('ABSPATH') || exit;

class Weels_Meta_Box
{
    public static function init(): void
    {
        add_action('add_meta_boxes', [__CLASS__, 'register']);
    }

    public static function register(): void
    {
        $screen_ids = ['shop_order', 'woocommerce_page_wc-orders'];
        foreach ($screen_ids as $screen) {
            add_meta_box(
                'weels-delivery',
                'Weels Delivery',
                [__CLASS__, 'render'],
                $screen,
                'side',
                'high'
            );
        }
    }

    public static function render($post_or_order): void
    {
        $order = ($post_or_order instanceof \WP_Post)
            ? wc_get_order($post_or_order->ID)
            : (method_exists($post_or_order, 'get_id') ? $post_or_order : wc_get_order($post_or_order->ID));

        if (!$order) {
            echo '<p>Order not found.</p>';
            return;
        }

        $order_id = $order->get_id();
        $task_id  = (int) $order->get_meta('_weels_task_id');
        $voided   = $order->get_meta('_weels_voided') === 'yes';
        $api      = new Weels_API();

        echo '<div class="weels-meta-box" data-order-id="' . esc_attr($order_id) . '">';

        if (!$api->has_key()) {
            echo '<p class="weels-notice weels-notice--warning">Weels API key not configured. <a href="' . esc_url(admin_url('admin.php?page=wc-settings&tab=shipping&section=weels_same_day')) . '">Configure &rarr;</a></p>';
            echo '</div>';
            return;
        }

        if ($task_id) {
            $tracking_url = $api->get_tracking_url($task_id);
            $label_url    = $api->get_label_url($task_id);

            echo '<div class="weels-status weels-status--shipped">';
            echo '<span class="weels-badge weels-badge--green">Shipped</span>';
            echo '<span class="weels-task-id">Shipment #' . esc_html($task_id) . '</span>';
            echo '</div>';

            echo '<div class="weels-actions">';
            echo '<a href="' . esc_url($label_url) . '" target="_blank" class="button weels-btn">View Label</a> ';
            echo '<a href="' . esc_url($tracking_url) . '" target="_blank" class="button weels-btn">Track</a> ';
            echo '<button type="button" class="button weels-btn weels-btn--danger" onclick="weelsVoid(' . esc_attr($order_id) . ')">Void</button>';
            echo '</div>';
        } elseif ($voided) {
            echo '<div class="weels-status">';
            echo '<span class="weels-badge weels-badge--grey">Voided</span>';
            echo '</div>';
            echo '<div class="weels-actions">';
            echo '<button type="button" class="button button-primary weels-btn" onclick="weelsGenerate(' . esc_attr($order_id) . ')">Regenerate Label</button>';
            echo '</div>';
        } else {
            echo '<div class="weels-status">';
            echo '<span class="weels-badge weels-badge--yellow">No label</span>';
            echo '</div>';
            echo '<div class="weels-actions">';
            echo '<button type="button" class="button button-primary weels-btn" id="weels-gen-btn" onclick="weelsGenerate(' . esc_attr($order_id) . ')">Generate Label</button>';
            echo '</div>';

            if (get_option('weels_auto_fulfill', 'no') === 'yes') {
                echo '<p class="weels-hint">Auto-fulfill is on. Labels are generated automatically when orders are paid.</p>';
            }
        }

        echo '<div id="weels-result" style="margin-top:8px;"></div>';
        echo '</div>';
    }
}
