<?php
/**
 * Weels Order Processing
 * Handles auto/manual label generation when orders change status.
 */

defined('ABSPATH') || exit;

class Weels_Order
{
    public static function init(): void
    {
        add_action('woocommerce_order_status_processing', [__CLASS__, 'on_order_processing']);
        add_action('wp_ajax_weels_generate_label', [__CLASS__, 'ajax_generate_label']);
        add_action('wp_ajax_weels_void_label', [__CLASS__, 'ajax_void_label']);
    }

    /**
     * Auto-fulfill on order status change to "processing" (paid).
     */
    public static function on_order_processing(int $order_id): void
    {
        if (get_option('weels_auto_fulfill', 'no') !== 'yes') return;

        $order = wc_get_order($order_id);
        if (!$order) return;

        if ($order->get_meta('_weels_task_id')) return;

        self::create_shipment($order);
    }

    /**
     * AJAX: Manual label generation from the order meta box.
     */
    public static function ajax_generate_label(): void
    {
        check_ajax_referer('weels_admin', 'nonce');

        if (!current_user_can('edit_shop_orders')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $order_id = (int) ($_POST['order_id'] ?? 0);
        $order = wc_get_order($order_id);

        if (!$order) {
            wp_send_json_error(['message' => 'Order not found']);
        }

        $existing = $order->get_meta('_weels_task_id');
        if ($existing) {
            $api = new Weels_API();
            wp_send_json_success([
                'task_id'      => (int) $existing,
                'tracking_url' => $api->get_tracking_url((int) $existing),
                'label_url'    => $api->get_label_url((int) $existing),
                'message'      => 'Label already exists.',
            ]);
        }

        $result = self::create_shipment($order);
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX: Void a label from the order meta box.
     */
    public static function ajax_void_label(): void
    {
        check_ajax_referer('weels_admin', 'nonce');

        if (!current_user_can('edit_shop_orders')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $order_id = (int) ($_POST['order_id'] ?? 0);
        $order = wc_get_order($order_id);

        if (!$order) {
            wp_send_json_error(['message' => 'Order not found']);
        }

        $task_id = (int) $order->get_meta('_weels_task_id');
        if (!$task_id) {
            wp_send_json_error(['message' => 'No shipment to void']);
        }

        $api = new Weels_API();
        $result = $api->void_shipment($task_id);

        if ($result['success']) {
            $order->delete_meta_data('_weels_task_id');
            $order->update_meta_data('_weels_voided', 'yes');
            $order->save();
            $order->add_order_note('Weels shipment #' . $task_id . ' voided.');
            wp_send_json_success(['message' => 'Shipment voided']);
        } else {
            wp_send_json_error(['message' => $result['error'] ?? 'Failed to void']);
        }
    }

    /**
     * Create a Weels shipment from a WooCommerce order.
     */
    public static function create_shipment(\WC_Order $order): array
    {
        $api = new Weels_API();
        if (!$api->has_key()) {
            return ['success' => false, 'error' => 'Weels API key not configured'];
        }

        $shipping = $order->get_address('shipping');
        $billing  = $order->get_address('billing');

        $dest_postal = $shipping['postcode'] ?: $billing['postcode'];
        if (!$dest_postal) {
            return ['success' => false, 'error' => 'No shipping postal code on order'];
        }

        // Build origin from store address
        $origin = [
            'origin_fname'      => get_option('blogname', ''),
            'origin_lname'      => '',
            'origin_company'    => get_option('blogname', ''),
            'origin_phone'      => '',
            'origin_email'      => get_option('admin_email', ''),
            'origin_address1'   => get_option('woocommerce_store_address', ''),
            'origin_address2'   => get_option('woocommerce_store_address_2', ''),
            'origin_city'       => get_option('woocommerce_store_city', ''),
            'origin_province'   => self::get_store_province(),
            'origin_postalcode' => get_option('woocommerce_store_postcode', ''),
            'origin_country'    => 'CA',
        ];

        // Build destination
        $dest_addr = !empty($shipping['address_1']) ? $shipping : $billing;
        $destination = [
            'destination_fname'      => $dest_addr['first_name'] ?? '',
            'destination_lname'      => $dest_addr['last_name'] ?? '',
            'destination_company'    => $dest_addr['company'] ?? '',
            'destination_phone'      => $order->get_billing_phone() ?: '',
            'destination_email'      => $order->get_billing_email() ?: '',
            'destination_address1'   => $dest_addr['address_1'] ?? '',
            'destination_address2'   => $dest_addr['address_2'] ?? '',
            'destination_city'       => $dest_addr['city'] ?? '',
            'destination_province'   => $dest_addr['state'] ?? '',
            'destination_postalcode' => $dest_addr['postcode'] ?? '',
            'destination_country'    => $dest_addr['country'] ?? 'CA',
        ];

        // Build packages
        $packages = self::build_packages($order);

        // Delivery date (same-day if before cutoff)
        $delivery_date = self::calculate_delivery_date();

        $data = array_merge($origin, $destination, [
            'order_num'            => $order->get_order_number(),
            'reference'            => '',
            'delivery_date'        => $delivery_date,
            'age_verification'     => (int) get_option('weels_age_verification', 0),
            'signature_required'   => get_option('weels_signature_required', 'no') === 'yes' ? 1 : 0,
            'packages'             => $packages,
        ]);

        $result = $api->create_shipment($data);

        if ($result['success']) {
            $task_id = $result['task_id'];

            $order->update_meta_data('_weels_task_id', $task_id);
            $order->update_meta_data('_weels_tracking_url', $result['tracking_url']);
            $order->delete_meta_data('_weels_voided');
            $order->save();

            $tracking_url = $result['tracking_url'];
            $order->add_order_note(
                sprintf('Weels shipment created. Tracking: <a href="%s" target="_blank">#%d</a>', esc_url($tracking_url), $task_id),
                false,
                true
            );

            // Customer-visible note
            $order->add_order_note(
                sprintf('Your order is being delivered by Weels same-day delivery. Track your package: <a href="%s" target="_blank">Track Shipment</a>', esc_url($tracking_url)),
                true
            );

            $result['tracking_url'] = $tracking_url;
            $result['label_url']    = $api->get_label_url($task_id);
        } else {
            $order->add_order_note('Weels shipment failed: ' . ($result['error'] ?? 'Unknown error'));
        }

        return $result;
    }

    private static function build_packages(\WC_Order $order): array
    {
        $default_weight = (float) get_option('weels_default_weight', 1);
        $default_length = (float) get_option('weels_default_length', 25);
        $default_width  = (float) get_option('weels_default_width', 20);
        $default_height = (float) get_option('weels_default_height', 15);

        $total_weight = 0;
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if (!$product) continue;

            $weight = (float) $product->get_weight();
            $qty    = $item->get_quantity();

            if ($weight > 0) {
                $wc_unit = get_option('woocommerce_weight_unit', 'kg');
                $weight_kg = self::convert_weight($weight, $wc_unit);
                $total_weight += $weight_kg * $qty;
            }
        }

        return [[
            'qty'    => 1,
            'weight' => $total_weight > 0 ? round($total_weight, 2) : $default_weight,
            'length' => $default_length,
            'width'  => $default_width,
            'height' => $default_height,
        ]];
    }

    private static function convert_weight(float $weight, string $from_unit): float
    {
        return match ($from_unit) {
            'g'   => $weight / 1000,
            'lbs' => $weight * 0.453592,
            'oz'  => $weight * 0.0283495,
            default => $weight,
        };
    }

    private static function calculate_delivery_date(): string
    {
        $tz = new \DateTimeZone('America/Toronto');
        $now = new \DateTime('now', $tz);

        if ((int) $now->format('H') >= 14) {
            $now->modify('+1 day');
        }

        $dow = (int) $now->format('N');
        if ($dow === 6) $now->modify('+2 days');
        elseif ($dow === 7) $now->modify('+1 day');

        return $now->format('Y-m-d');
    }

    private static function get_store_province(): string
    {
        $location = get_option('woocommerce_default_country', 'CA:ON');
        $parts = explode(':', $location);
        return $parts[1] ?? 'ON';
    }
}
