<?php
/**
 * Weels Settings
 * Adds a settings section under WooCommerce > Settings > Shipping.
 */

defined('ABSPATH') || exit;

class Weels_Settings
{
    public static function init(): void
    {
        add_filter('woocommerce_get_sections_shipping', [__CLASS__, 'add_section']);
        add_filter('woocommerce_get_settings_shipping', [__CLASS__, 'get_settings'], 10, 2);
        add_action('woocommerce_admin_field_weels_api_status', [__CLASS__, 'render_api_status']);
    }

    public static function add_section(array $sections): array
    {
        $sections['weels_same_day'] = __('Weels Same-Day Delivery', 'weels-same-day-delivery');
        return $sections;
    }

    public static function get_settings(array $settings, string $current_section): array
    {
        if ($current_section !== 'weels_same_day') return $settings;

        return [
            [
                'title' => __('Weels Same-Day Delivery', 'weels-same-day-delivery'),
                'type'  => 'title',
                'desc'  => __('Connect your Weels account to enable same-day delivery for your WooCommerce store. <a href="https://www.weels.ca/woocommerce" target="_blank">Setup guide &rarr;</a>', 'weels-same-day-delivery'),
                'id'    => 'weels_settings',
            ],
            [
                'title'    => __('API Key', 'weels-same-day-delivery'),
                'desc'     => __('Your Weels API key. Find it at <a href="https://www.weels.ca/account" target="_blank">weels.ca/account</a> &rarr; Organization &rarr; API & Integrations.', 'weels-same-day-delivery'),
                'id'       => 'weels_api_key',
                'type'     => 'password',
                'css'      => 'min-width:400px;',
                'desc_tip' => false,
            ],
            [
                'type' => 'weels_api_status',
                'id'   => 'weels_api_status',
            ],
            [
                'title'   => __('Auto-Fulfill Orders', 'weels-same-day-delivery'),
                'desc'    => __('Automatically generate a shipping label when an order is paid', 'weels-same-day-delivery'),
                'id'      => 'weels_auto_fulfill',
                'type'    => 'checkbox',
                'default' => 'no',
            ],
            [
                'type' => 'sectionend',
                'id'   => 'weels_settings',
            ],
            [
                'title' => __('Default Package Dimensions', 'weels-same-day-delivery'),
                'type'  => 'title',
                'desc'  => __('Used when products don\'t have weight or dimensions set.', 'weels-same-day-delivery'),
                'id'    => 'weels_package_settings',
            ],
            [
                'title'   => __('Weight (kg)', 'weels-same-day-delivery'),
                'id'      => 'weels_default_weight',
                'type'    => 'number',
                'default' => '1',
                'css'     => 'width:100px;',
                'custom_attributes' => ['step' => '0.1', 'min' => '0'],
            ],
            [
                'title'   => __('Length (cm)', 'weels-same-day-delivery'),
                'id'      => 'weels_default_length',
                'type'    => 'number',
                'default' => '25',
                'css'     => 'width:100px;',
                'custom_attributes' => ['step' => '0.1', 'min' => '0'],
            ],
            [
                'title'   => __('Width (cm)', 'weels-same-day-delivery'),
                'id'      => 'weels_default_width',
                'type'    => 'number',
                'default' => '20',
                'css'     => 'width:100px;',
                'custom_attributes' => ['step' => '0.1', 'min' => '0'],
            ],
            [
                'title'   => __('Height (cm)', 'weels-same-day-delivery'),
                'id'      => 'weels_default_height',
                'type'    => 'number',
                'default' => '15',
                'css'     => 'width:100px;',
                'custom_attributes' => ['step' => '0.1', 'min' => '0'],
            ],
            [
                'type' => 'sectionend',
                'id'   => 'weels_package_settings',
            ],
            [
                'title' => __('Delivery Options', 'weels-same-day-delivery'),
                'type'  => 'title',
                'id'    => 'weels_delivery_settings',
            ],
            [
                'title'   => __('Age Verification', 'weels-same-day-delivery'),
                'id'      => 'weels_age_verification',
                'type'    => 'select',
                'default' => '0',
                'options' => [
                    '0'  => __('None', 'weels-same-day-delivery'),
                    '18' => __('18+', 'weels-same-day-delivery'),
                    '19' => __('19+', 'weels-same-day-delivery'),
                    '21' => __('21+', 'weels-same-day-delivery'),
                ],
            ],
            [
                'title'   => __('Signature Required', 'weels-same-day-delivery'),
                'desc'    => __('Require a digital signature on delivery', 'weels-same-day-delivery'),
                'id'      => 'weels_signature_required',
                'type'    => 'checkbox',
                'default' => 'no',
            ],
            [
                'type' => 'sectionend',
                'id'   => 'weels_delivery_settings',
            ],
        ];
    }

    public static function render_api_status(): void
    {
        $api_key = get_option('weels_api_key', '');
        if (!$api_key) {
            echo '<tr><th></th><td><span style="color:#999;">Enter your API key and save to validate.</span></td></tr>';
            return;
        }

        $api = new Weels_API($api_key);
        $result = $api->validate_key();

        if ($result['valid']) {
            echo '<tr><th></th><td><span style="color:#46b450;font-weight:600;">&#10003; API key verified — connected to Weels</span></td></tr>';
        } else {
            echo '<tr><th></th><td><span style="color:#dc3232;font-weight:600;">&#10007; ' . esc_html($result['error'] ?? 'Invalid API key') . '</span></td></tr>';
        }
    }
}
