<?php
/**
 * Weels Shipping Method
 * WooCommerce shipping method that fetches live rates from the Weels API.
 */

defined('ABSPATH') || exit;

class Weels_Shipping_Method extends WC_Shipping_Method
{
    public function __construct(int $instance_id = 0)
    {
        $this->id                 = 'weels_same_day';
        $this->instance_id        = absint($instance_id);
        $this->method_title       = __('Weels Same-Day Delivery', 'weels-same-day-delivery');
        $this->method_description = __('Same-day delivery across the Greater Toronto Area via Weels.', 'weels-same-day-delivery');
        $this->supports           = ['shipping-zones', 'instance-settings'];

        $this->init();
    }

    private function init(): void
    {
        $this->init_form_fields();
        $this->init_settings();

        $this->title   = $this->get_option('title', 'Weels Same-Day Delivery');
        $this->enabled = $this->get_option('enabled', 'yes');

        add_action('woocommerce_update_options_shipping_' . $this->id, [$this, 'process_admin_options']);
    }

    public function init_form_fields(): void
    {
        $this->instance_form_fields = [
            'title' => [
                'title'   => __('Method Title', 'weels-same-day-delivery'),
                'type'    => 'text',
                'default' => 'Weels Same-Day Delivery',
                'desc_tip' => true,
                'description' => __('The name shown to customers at checkout.', 'weels-same-day-delivery'),
            ],
        ];
    }

    public function is_available($package): bool
    {
        $api = new Weels_API();
        if (!$api->has_key()) return false;

        $country = $package['destination']['country'] ?? '';
        if ($country && $country !== 'CA') return false;

        $postal = $package['destination']['postcode'] ?? '';
        if (!$postal) return false;

        return true;
    }

    public function calculate_shipping($package = []): void
    {
        $api = new Weels_API();
        if (!$api->has_key()) return;

        $postal = $package['destination']['postcode'] ?? '';
        if (!$postal) return;

        $qty = 0;
        foreach ($package['contents'] as $item) {
            $qty += $item['quantity'];
        }
        if ($qty < 1) $qty = 1;

        $age  = (int) get_option('weels_age_verification', 0);
        $sig  = (int) get_option('weels_signature_required', 0);

        $rates = $api->get_rates($postal, 1, [
            'age_verification' => $age,
            'signature'        => $sig,
        ]);

        if (!$rates || empty($rates['available'])) return;

        $total = (float) ($rates['rate']['total'] ?? 0);
        if ($total <= 0) return;

        $delivery_days = (int) ($rates['delivery_days'] ?? 0);
        $label = $this->title;
        if ($delivery_days === 0) {
            $label .= ' (Today)';
        } elseif ($delivery_days === 1) {
            $label .= ' (Tomorrow)';
        }

        $this->add_rate([
            'id'       => $this->get_rate_id(),
            'label'    => $label,
            'cost'     => $total,
            'calc_tax' => 'per_order',
            'meta_data' => [
                'weels_city'     => $rates['city'] ?? '',
                'weels_province' => $rates['province'] ?? '',
            ],
        ]);
    }
}
