<?php
/**
 * Plugin Name: Weels Same-Day Delivery
 * Plugin URI: https://www.weels.ca/woocommerce
 * Description: Same-day delivery for WooCommerce. Generate shipping labels, sync tracking, and show live delivery rates at checkout across the Greater Toronto Area.
 * Version: 1.0.0
 * Author: Weels Inc.
 * Author URI: https://www.weels.ca
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: weels-same-day-delivery
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * WC requires at least: 7.0
 * WC tested up to: 9.0
 */

defined('ABSPATH') || exit;

define('WEELS_VERSION', '1.0.0');
define('WEELS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WEELS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WEELS_API_BASE', 'https://www.weels.ca/api');

require_once WEELS_PLUGIN_DIR . 'includes/class-weels-api.php';
require_once WEELS_PLUGIN_DIR . 'includes/class-weels-settings.php';
require_once WEELS_PLUGIN_DIR . 'includes/class-weels-order.php';
require_once WEELS_PLUGIN_DIR . 'includes/class-weels-meta-box.php';

final class Weels_Same_Day_Delivery
{
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('plugins_loaded', [$this, 'init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'settings_link']);
    }

    public function init(): void
    {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-error"><p><strong>Weels Same-Day Delivery</strong> requires WooCommerce to be installed and active.</p></div>';
            });
            return;
        }

        add_filter('woocommerce_shipping_methods', [$this, 'register_shipping_method']);

        Weels_Settings::init();
        Weels_Order::init();
        Weels_Meta_Box::init();
    }

    public function register_shipping_method(array $methods): array
    {
        require_once WEELS_PLUGIN_DIR . 'includes/class-weels-shipping.php';
        $methods['weels_same_day'] = 'Weels_Shipping_Method';
        return $methods;
    }

    public function enqueue_admin_assets(string $hook): void
    {
        $screen = get_current_screen();
        if (!$screen) return;

        $is_order_page = in_array($screen->id, ['shop_order', 'woocommerce_page_wc-orders'], true)
            || ($screen->post_type === 'shop_order');
        $is_settings_page = $screen->id === 'woocommerce_page_wc-settings';

        if ($is_order_page || $is_settings_page) {
            wp_enqueue_style('weels-admin', WEELS_PLUGIN_URL . 'assets/weels-admin.css', [], WEELS_VERSION);
            wp_enqueue_script('weels-admin', WEELS_PLUGIN_URL . 'assets/weels-admin.js', ['jquery'], WEELS_VERSION, true);
            wp_localize_script('weels-admin', 'weelsAdmin', [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('weels_admin'),
            ]);
        }
    }

    public function settings_link(array $links): array
    {
        $url = admin_url('admin.php?page=wc-settings&tab=shipping&section=weels_same_day');
        array_unshift($links, '<a href="' . esc_url($url) . '">Settings</a>');
        return $links;
    }
}

Weels_Same_Day_Delivery::instance();
